'use strict';

/* --------------------------------------------------------------
 editor_instances.js 2016-09-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.editor_instances = jse.libs.editor_instances || {};

/**
 * ## Editor Instances Library
 *
 * This library provides a common API for editor widget instances manipulation.
 *
 * @module Admin/Libs/editor_instances
 * @exports jse.libs.editor_instances
 */
(function (exports) {

    'use strict';

    /**
     * Editor construct methods
     *
     * @type {Object}
     */

    var constructors = {
        ckeditor: function ckeditor($textarea) {
            var tags = ['p', 'div', 'script', 'style', 'form'];

            var formattingRules = {
                indent: true,
                breakBeforeOpen: true,
                breakAfterOpen: true,
                breakBeforeClose: true,
                breakAfterClose: true
            };

            var config = {
                filebrowserBrowseUrl: 'includes/ckeditor/filemanager/index.html',
                baseHref: jse.core.config.get('appUrl') + '/admin',
                enterMode: CKEDITOR.ENTER_BR,
                shiftEnterMode: CKEDITOR.ENTER_P,
                language: jse.core.config.get('languageCode'),
                useRelPath: true,
                on: {
                    instanceReady: function instanceReady(event) {
                        var _iteratorNormalCompletion = true;
                        var _didIteratorError = false;
                        var _iteratorError = undefined;

                        try {
                            for (var _iterator = tags[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                                var tag = _step.value;

                                this.dataProcessor.writer.setRules(tag, formattingRules);
                            }
                        } catch (err) {
                            _didIteratorError = true;
                            _iteratorError = err;
                        } finally {
                            try {
                                if (!_iteratorNormalCompletion && _iterator.return) {
                                    _iterator.return();
                                }
                            } finally {
                                if (_didIteratorError) {
                                    throw _iteratorError;
                                }
                            }
                        }
                    }
                }
            };

            var name = $textarea.attr('name');
            CKEDITOR.replace(name, config);
            return CKEDITOR.instances[name];
        },
        codemirror: function codemirror($textarea) {
            var config = {
                mode: 'htmlmixed',
                lineNumbers: true,
                lineWrapping: true
            };

            return CodeMirror.fromTextArea($textarea[0], config);
        }
    };

    /**
     * Editor destruct methods
     *
     * @type {Object}
     */
    var destructors = {
        ckeditor: function ckeditor($textarea) {
            var name = $textarea.attr('name');
            CKEDITOR.instances[name].destroy();
        },
        codemirror: function codemirror($textarea) {
            var instance = $textarea.siblings('.CodeMirror')[0].CodeMirror;
            instance.toTextArea();
        }
    };

    /**
     * Editor instance methods
     *
     * @type {Object}
     */
    var getInstance = {
        ckeditor: function ckeditor($textarea) {
            return CKEDITOR.instances[$textarea.attr('name')];
        },
        codemirror: function codemirror($textarea) {
            return $textarea.siblings('.CodeMirror')[0].CodeMirror;
        }
    };

    /**
     * Create new editor instance.
     *
     * @param {jQuery} $textarea Textarea selector to be modified.
     * @param {String} type Provide a type that is supported by the widget.
     *
     * @return {CKEditor|CodeMirror} Returns the create editor instance.
     */
    exports.create = function ($textarea, type) {
        if (constructors[type] === undefined) {
            throw new Error('Provided editor type is not supported: ' + type);
        }

        var instance = constructors[type]($textarea);
        instance.type = type;

        return instance;
    };

    /**
     * Switch to a new editor type.
     *
     * @param {jQuery} $textarea Textarea selector to be modified.
     * @param {String} currentType Provide the current editor type.
     * @param {String} newType Provide the new editor type.
     *
     * @return {CKEditor|CodeMirror} Returns the new editor instance.
     */
    exports.switch = function ($textarea, currentType, newType) {
        if (destructors[currentType] === undefined) {
            throw new Error('Provided editor type is not supported: ' + currentType);
        }

        destructors[currentType]($textarea);

        return exports.create($textarea, newType);
    };

    /**
     * Destroy an existing editor instance.
     *
     * @param {jQuery} $textarea Textarea selector to be destroyed.
     * @param {String} customInitEventType Optional (''), if the editor was initialized with a custom init-event-type,
     * then this must be unbound by editor destroy.
     */
    exports.destroy = function ($textarea) {
        var customInitEventType = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : '';

        if (customInitEventType && customInitEventType !== 'JSENGINE_INIT_FINISHED') {
            $(window).off(customInitEventType);
        }

        var type = $textarea.data('editorType');

        if (destructors[type] === undefined) {
            throw new Error('Provided editor type is not supported: ' + type);
        }

        destructors[type]($textarea);
    };

    /**
     * Get an editor instance.
     *
     * @param {jQuery} $textarea Textarea selector from which the instance will be retrieved.
     *
     * @return {CKEditor|CodeMirror} Returns the corresponding editor instance.
     */
    exports.getInstance = function ($textarea) {
        var type = $textarea.data('editorType');

        if (destructors[type] === undefined) {
            throw new Error('Provided editor type is not supported: ' + type);
        }

        return getInstance[type]($textarea);
    };
})(jse.libs.editor_instances);
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
